package gov.va.med.mhv.getcare.validator;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.getcare.common.dto.HealthCareProviderDTO;
import gov.va.med.mhv.getcare.enums.ProviderTypeEnumeration;
import gov.va.med.mhv.getcare.model.HealthCareProvider;
import gov.va.med.mhv.getcare.repository.HealthCareProviderRepository;
import gov.va.med.mhv.getcare.util.MessagesUtil;

@Component
public class HealthCareProviderValidator  extends Validator{
	
	@Autowired
	private HealthCareProviderRepository healthCareProviderRepository;
	
	@Autowired
	private MessagesUtil messages;
	
	private static final int COMMENTS_MAX_LENGTH = 255;
	private static final int OTHERCLINICIAN_MAX_LENGTH = 25;
	private static final int FIRSTNAME_MIN_LENGTH = 2;
	private static final int FIRSTNAME_MAX_LENGTH = 50;
	private static final int LASTNAME_MIN_LENGTH = 2;
	private static final int LASTNAME_MAX_LENGTH = 50;
	private static final int WORK_PHONE_EXT_MAX_LENGTH = 255;
	private static final int WORK_PHONE_MAX_LENGTH = 255;
	private static final int EMAIL_MAX_LENGTH = 255;
	
	public void providerValidations(ResponseUtil response, HealthCareProviderDTO providerDTO) {
		fieldValidation(response, providerDTO);
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, providerDTO);
		}
	}

	private void complexValidation(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		validateSinglePrimaryProvider(response, providerDTO);
	}

	private void validateSinglePrimaryProvider(ResponseUtil response, HealthCareProviderDTO providerDTO) {
		
		if(providerDTO.getProviderType().equals(ProviderTypeEnumeration.P.toString())){
			List<HealthCareProvider> providers=healthCareProviderRepository.findPrimaryHealthCareProvider(providerDTO.getUserProfileId());
			if(providers != null && providers.size() > 0){
				for(HealthCareProvider provider: providers){
					if(providerDTO.getHealthCareProviderId() != null){
					   if(providerDTO.getHealthCareProviderId().longValue() != provider.getHealthCareProviderId().longValue()){
						   response.getValidationErrors().put("invalid provider",messages.getInvalidPrimaryProvider());
						   break;	
					   }
					}else{
					    response.getValidationErrors().put("invalid provider",messages.getInvalidPrimaryProvider());
						break;
					}
				}
			}
		}
	}

	private void fieldValidation(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		validateComments(response,providerDTO);
		validateOtherClinician(response,providerDTO);
		validateContactInfoWorkPhoneExtension(response,providerDTO);
		validateContactInfoEmail(response,providerDTO);
		validateProviderType(response,providerDTO);
		validateContactInfoWorkPhone(response,providerDTO);
		validateFirstName(response,providerDTO);
		validateLastName(response,providerDTO);
		validateUserProfile(response,providerDTO);
	}

	private void validateUserProfile(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		if( providerDTO.getUserProfileId() == null ) {
			response.getValidationErrors().put("userProfile", "userProfile is required");
			return;
		}
		boolean valid= isUserProfileValid(providerDTO.getUserProfileId() );
		if ( !valid) {
			response.getValidationErrors().put("userProfile", "userProfile is Invalid");
		}
	}

	private void validateLastName(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getLastName();
		if ( value == null ) {
			Validations.validateRequired("Provider Last Name", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMinLength("Provider Last Name",value, LASTNAME_MIN_LENGTH, response.getValidationErrors());
		Validations.validateMaxLength("Provider Last Name",value, LASTNAME_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateFirstName(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getFirstName();
		if ( value == null ) {
			Validations.validateRequired("Provider First Name", value, response.getValidationErrors());
			return;
		}	
		Validations.validateMinLength("Provider First Name",value, FIRSTNAME_MIN_LENGTH, response.getValidationErrors());
		Validations.validateMaxLength("Provider First Name",value, FIRSTNAME_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateContactInfoWorkPhone(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getWorkPhone();
		if ( value == null ) {
			Validations.validateRequired("Phone Number", value, response.getValidationErrors());
			return;
		}						
		Validations.validateMaxLength("Phone Number",value, WORK_PHONE_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateProviderType(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getProviderType();
		if( value == null ) {
			return;
		}		
		Validations.validateEnumeration("Provider Type", value, ProviderTypeEnumeration.values(), response.getValidationErrors());
	}

	private void validateContactInfoEmail(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getEmailAddress();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Email Address",value, EMAIL_MAX_LENGTH, response.getValidationErrors());
		Validations.validateEmail("Email Address", value, response.getValidationErrors());
	}

	private void validateContactInfoWorkPhoneExtension(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getWorkPhoneExt();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Work Phone Extension",value, WORK_PHONE_EXT_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateOtherClinician(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getOtherClinician();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Other Clinician",value, OTHERCLINICIAN_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateComments(ResponseUtil response,HealthCareProviderDTO providerDTO) {
		String value = providerDTO.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
	}
}
